<#
Title: Boxafe Certificate Generator
Description: This script will generate a certificate and manifest file to authorize Boxafe application in the azure portal.
Author: Qnap Systems Inc
Email: marketing.tw@qnap.com
Last modified on: May 3, 2021
#>

[CmdletBinding()]
param(
    $AzureEnvironmentName
)

Function SystemCheck() {
    # Verify script execution environment compatibility
    if ([System.Environment]::OSVersion.Version.Major -lt 10) {
        Write-Host "This PowerShell script file can only run on Windows 10 and Windows Server 2016 or later." -ForegroundColor Red -BackgroundColor white
        Exit
    }

    # Set SecurityProtocol to be compatible for W2016 server
    [Net.ServicePointManager]::SecurityProtocol = 'Ssl3, Tls, Tls11, Tls12'

    if ($null -eq (Get-Module -ListAvailable -Name "AzureAD")) {
        Install-Module "AzureAD" -Scope CurrentUser
    }
    Import-Module AzureAD
}

Function AddPermission($requiredAccess, `
        $exposedPermissions, [string]$requiredAccesses, [string]$permissionType) {
    foreach ($permission in $requiredAccesses.Trim().Split("|")) {
        foreach ($exposedPermission in $exposedPermissions) {
            if ($exposedPermission.Value -eq $permission) {
                $resourceAccess = New-Object Microsoft.Open.AzureAD.Model.ResourceAccess
                $resourceAccess.Type = $permissionType
                $resourceAccess.Id = $exposedPermission.Id
                $requiredAccess.ResourceAccess.Add($resourceAccess)
            }
        }
    }
}

Function SetPermissions([string] $appDisplayName, [string] $delegatedPermissions, [string]$appPermissions, $servicePrincipal) {
    if ($servicePrincipal) {
        $sp = $servicePrincipal
    }
    else {
        $sp = Get-AzureADServicePrincipal -Filter "DisplayName eq '$appDisplayName'"
    }
    $appid = $sp.AppId
    $requiredAccess = New-Object Microsoft.Open.AzureAD.Model.RequiredResourceAccess
    $requiredAccess.ResourceAppId = $appid
    $requiredAccess.ResourceAccess = New-Object System.Collections.Generic.List[Microsoft.Open.AzureAD.Model.ResourceAccess]

    if ($delegatedPermissions) {
        AddPermission $requiredAccess -exposedPermissions $sp.Oauth2Permissions -requiredAccesses $delegatedPermissions -permissionType "Scope"
    }

    if ($appPermissions) {
        AddPermission $requiredAccess -exposedPermissions $sp.AppRoles -requiredAccesses $appPermissions -permissionType "Role"
    }
    return $requiredAccess
}

Function GenerateFileName([string] $fileName, [string] $extension) {
    # Generate new file, also will take file not get overridden
    $location = Get-Location
    $path = "$($location)\$($fileName).$($extension)"
    $count = 1

    while (Test-Path $path -PathType Leaf) {
        $path = "$($location)\$($fileName) ($($count)).$($extension)"
        $count += 1
    }
    return $path
}

Function ReadPassword([Int16] $count) {
    # Get the password from the user input
    $password = Read-Host 'Please enter a password to encrypt the certificate: ' -AsSecureString
    if ($password.Length -eq 0) {
        Write-Host "[ERROR] Password is mandatory." -ForegroundColor Red -BackgroundColor white
        $count += 1
        if ($count -eq 3) {
            Write-Host "Max retry exceeded. Exiting..."
            Exit
        }
        $password = ReadPassword($count)
    }
    return $password
}
Function GenerateApplication {

    # Connect to Azure AD
    $creds = Connect-AzureAD -AzureEnvironmentName $AzureEnvironmentName
    if (!$creds) {
        Write-Host "[ERROR] Failed to connect azure AD. Try after some time" -ForegroundColor Red -BackgroundColor white
        Exit
    }
    $tenantId = $creds.Tenant.Id

    $appName = "Boxafe Backup Solution"

    # Get the user running the script
    $user = Get-AzureADUser -ObjectId $creds.Account.Id

    # Create the client AAD application
    Write-Host "[INFO] Registering app ($appName) inside Azure AD..."
    $application = New-AzureADApplication -DisplayName $appName

    # Generate self signed certificate
    $certificate = New-SelfSignedCertificate -Subject CN=$appName `
        -CertStoreLocation "Cert:\CurrentUser\My" `
        -KeyExportPolicy Exportable `
        -NotAfter (Get-Date).AddYears(100) `
        -KeySpec Signature

    $pfxFilePath = GenerateFileName -fileName "Certificate" -extension "pfx"
    Export-PfxCertificate -Cert $certificate -FilePath $pfxFilePath -Password $password > $null
    # Generate key credentials
    $certKeyId = [Guid]::NewGuid()
    $certBase64Value = [System.Convert]::ToBase64String($certificate.GetRawCertData())
    $certBase64Thumbprint = [System.Convert]::ToBase64String($certificate.GetCertHash())
    # Add certificate into azure app
    New-AzureADApplicationKeyCredential -ObjectId $application.ObjectId `
        -CustomKeyIdentifier $certBase64Thumbprint `
        -Type AsymmetricX509Cert `
        -Usage Verify `
        -Value $certBase64Value `
        -StartDate $certificate.NotBefore `
        -EndDate $certificate.NotAfter
    # Set current user as application owner
    $owner = Get-AzureADApplicationOwner -ObjectId $application.ObjectId
    if ($null -eq $owner) {
        Add-AzureADApplicationOwner -ObjectId $application.ObjectId -RefObjectId $user.ObjectId
        Write-Host "[INFO] '$($user.UserPrincipalName)' added as an application owner to '$($appName)' app"
    }

    $credentials = @{ type      = "AsymmetricX509Cert"; `
            usage               = "Verify"; `
            keyId               = "$($certKeyId)"; `
            customKeyIdentifier = "$($certBase64Thumbprint)"; `
            value               = "$($certBase64Value)";
    } | ConvertTo-Json -Compress

    # Export key credentials
    $credentialsFilePath = GenerateFileName -fileName "KeyCredentials" -extension "txt"
    Set-Content -Path $credentialsFilePath -Value $credentials -Force

    Write-Host "[INFO] ($appName) application registered."
    # URL of the AAD application in the Azure portal
    $path = switch ($AzureEnvironmentName) {
        "AzureCloud" { "https://portal.azure.com/" }
        "AzureChinaCloud" { "https://portal.azure.cn/" }
        "AzureGermanyCloud" { "https://portal.microsoftazure.de/" }
        Default { "https://portal.azure.com/" }
    }
    $azureUrl = "$($path )#blade/Microsoft_AAD_RegisteredApps/ApplicationMenuBlade/CallAnAPI/appId/$($application.AppId)/objectId/$($application.ObjectId)/isMSAApp/"

    $requiredResourceAccess = New-Object System.Collections.Generic.List[Microsoft.Open.AzureAD.Model.RequiredResourceAccess]

    # Set required API permissions to the app created for Boxafe
    Write-Host "[INFO] Preparing API permission list for Boxafe..."
    $permissions = SetPermissions -appDisplayName "Microsoft Graph" `
        -appPermissions "Mail.ReadWrite|User.ReadWrite.All|Group.ReadWrite.All|Directory.ReadWrite.All|Contacts.ReadWrite|Files.ReadWrite.All|Calendars.ReadWrite|Sites.FullControl.All|Sites.ReadWrite.All|ChannelMessage.Read.All|TeamMember.ReadWrite.All|ChannelMember.ReadWrite.All";
    $requiredResourceAccess.Add($permissions)

    $permissions = SetPermissions -appDisplayName "Office 365 Exchange Online" `
        -appPermissions "full_access_as_app|Calendars.ReadWrite.All";
    $requiredResourceAccess.Add($permissions)

    $permissions = SetPermissions -appDisplayName "Office 365 SharePoint Online" `
        -appPermissions "Sites.FullControl.All";
    $requiredResourceAccess.Add($permissions)

    Write-Host "[INFO] Setting Microsoft Graph, Microsoft 365 Exchange Online and Microsoft 365 SharePoint Online API permissions for Boxafe..."
    Set-AzureADApplication -ObjectId $application.ObjectId -RequiredResourceAccess $requiredResourceAccess

    Write-Host "@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@"
    Write-Host "@                                                                                 @"
    Write-Host "@   Congratulations! Your Azure application has been registered successfully.     @"
    Write-Host "@                                                                                 @"
    Write-Host "@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@"
    Write-Host ""
    Write-Host "Please copy following details and certificate file and store at the secure place."
    Write-Host ""
    Write-Host "Tenant Id:"
    Write-Host "$($tenantId)" -ForegroundColor Green
    Write-Host "Application Id (Client Id):"
    Write-Host "$($application.AppId)" -ForegroundColor Green
    Write-Host "PFX Certificate file(Keep securely):"
    Write-Host "$($pfxFilePath)" -ForegroundColor Green
    Write-Host "Key credentials: (Required to update manifest details in the azure portal)"
    Write-Host "$($credentialsFilePath)" -ForegroundColor Green
    Write-Host ""
    Write-Host ""
    Write-Host "Important: Please open following URL in your browser and grant admin access to the application: ($appName)" -ForegroundColor Red
    Write-Host "$azureUrl" -ForegroundColor Yellow
    Write-Host ""
}

SystemCheck
$password = ReadPassword(0)
GenerateApplication -password $password


# SIG # Begin signature block
# MIIZcQYJKoZIhvcNAQcCoIIZYjCCGV4CAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQUPNfRSQ9FL4UEN9yQ28Jwz0EZ
# 1lugghRyMIIE2jCCA8KgAwIBAgIQcihhzDecMy1dmXpidSGfHzANBgkqhkiG9w0B
# AQsFADB/MQswCQYDVQQGEwJVUzEdMBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRp
# b24xHzAdBgNVBAsTFlN5bWFudGVjIFRydXN0IE5ldHdvcmsxMDAuBgNVBAMTJ1N5
# bWFudGVjIENsYXNzIDMgU0hBMjU2IENvZGUgU2lnbmluZyBDQTAeFw0xOTAxMTgw
# MDAwMDBaFw0yMjAzMjgyMzU5NTlaMHIxCzAJBgNVBAYTAlRXMQ8wDQYDVQQIDAZU
# YWl3YW4xGDAWBgNVBAcMD05ldyBUYWlwZWkgQ2l0eTEbMBkGA1UECgwSUU5BUCBT
# eXN0ZW1zLCBJbmMuMRswGQYDVQQDDBJRTkFQIFN5c3RlbXMsIEluYy4wggEiMA0G
# CSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCWxfXrj79U6X/vE2lSwlTTj5Aq1BDb
# PUKIK1Ge/M9vREg9YUpws0L3jt3yMaJF8Du59EiC4ZXlLmVMYkq4OCuJ6tSD3e8z
# 9+wZ3ZH/7mBNQkCzZV/3y/4ytXfyl3LrF2c3HT048O+sqbuPhJlBHxsLJAHrczT4
# mgR37nGOgUTcN1yrFreeAYnk3D1YZCssUbas4hjTaYzU42giaQYd9Yu9PXLzd2eM
# 6hvVOXAV8Cz4qN/2/S3foxwV0g5JlPsc+lGtTEUstjK9nGieW6aEG0+B7V2QLArl
# ufgYVxW6+1epeCPXyLMrPo0RJGCZdVgzmqCooERR2U0kHpmzu0kfd0BlAgMBAAGj
# ggFdMIIBWTAJBgNVHRMEAjAAMA4GA1UdDwEB/wQEAwIHgDArBgNVHR8EJDAiMCCg
# HqAchhpodHRwOi8vc3Yuc3ltY2IuY29tL3N2LmNybDBhBgNVHSAEWjBYMFYGBmeB
# DAEEATBMMCMGCCsGAQUFBwIBFhdodHRwczovL2Quc3ltY2IuY29tL2NwczAlBggr
# BgEFBQcCAjAZDBdodHRwczovL2Quc3ltY2IuY29tL3JwYTATBgNVHSUEDDAKBggr
# BgEFBQcDAzBXBggrBgEFBQcBAQRLMEkwHwYIKwYBBQUHMAGGE2h0dHA6Ly9zdi5z
# eW1jZC5jb20wJgYIKwYBBQUHMAKGGmh0dHA6Ly9zdi5zeW1jYi5jb20vc3YuY3J0
# MB8GA1UdIwQYMBaAFJY7U/B5M5evfYPvLivMyreGHnJmMB0GA1UdDgQWBBQzWCZm
# hrC4vlEMLpiwYi/ROO6OTzANBgkqhkiG9w0BAQsFAAOCAQEATWzRDS2qC35Cmqjd
# TmV81peFYQCsYipHsXxc0HQgUn6p9wvysU7/LZnga1abzi19fIaIHjZ6e0yShc2a
# nZk1SkS8vKGxcP7bepDLiUB+q3iU+30uCayBz4+Pql9a69kAPhxfzwOo6R8Y0utj
# DVYhDrdxCGZlnOJffsPZVO1NJS3e1arJYx9p2tnnf1DNXvOnh/mlUA+38moa+Q7c
# Y/K4lOPu03T7AQUHnilfuihqKgS/C9A60N6TqT29BRcbtyArpyKqiY8gY+G61t/Z
# Gvux00uMG6R3dOz5/QdTsZbbMINqSRgC898oXZbi1auAD+vs7Je8SfBrBzCpqjiN
# g/ze8TCCBP4wggPmoAMCAQICEA1CSuC+Ooj/YEAhzhQA8N0wDQYJKoZIhvcNAQEL
# BQAwcjELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UE
# CxMQd3d3LmRpZ2ljZXJ0LmNvbTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1
# cmVkIElEIFRpbWVzdGFtcGluZyBDQTAeFw0yMTAxMDEwMDAwMDBaFw0zMTAxMDYw
# MDAwMDBaMEgxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjEg
# MB4GA1UEAxMXRGlnaUNlcnQgVGltZXN0YW1wIDIwMjEwggEiMA0GCSqGSIb3DQEB
# AQUAA4IBDwAwggEKAoIBAQDC5mGEZ8WK9Q0IpEXKY2tR1zoRQr0KdXVNlLQMULUm
# EP4dyG+RawyW5xpcSO9E5b+bYc0VkWJauP9nC5xj/TZqgfop+N0rcIXeAhjzeG28
# ffnHbQk9vmp2h+mKvfiEXR52yeTGdnY6U9HR01o2j8aj4S8bOrdh1nPsTm0zinxd
# RS1LsVDmQTo3VobckyON91Al6GTm3dOPL1e1hyDrDo4s1SPa9E14RuMDgzEpSlwM
# MYpKjIjF9zBa+RSvFV9sQ0kJ/SYjU/aNY+gaq1uxHTDCm2mCtNv8VlS8H6GHq756
# WwogL0sJyZWnjbL61mOLTqVyHO6fegFz+BnW/g1JhL0BAgMBAAGjggG4MIIBtDAO
# BgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAKBggrBgEF
# BQcDCDBBBgNVHSAEOjA4MDYGCWCGSAGG/WwHATApMCcGCCsGAQUFBwIBFhtodHRw
# Oi8vd3d3LmRpZ2ljZXJ0LmNvbS9DUFMwHwYDVR0jBBgwFoAU9LbhIB3+Ka7S5GGl
# sqIlssgXNW4wHQYDVR0OBBYEFDZEho6kurBmvrwoLR1ENt3janq8MHEGA1UdHwRq
# MGgwMqAwoC6GLGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9zaGEyLWFzc3VyZWQt
# dHMuY3JsMDKgMKAuhixodHRwOi8vY3JsNC5kaWdpY2VydC5jb20vc2hhMi1hc3N1
# cmVkLXRzLmNybDCBhQYIKwYBBQUHAQEEeTB3MCQGCCsGAQUFBzABhhhodHRwOi8v
# b2NzcC5kaWdpY2VydC5jb20wTwYIKwYBBQUHMAKGQ2h0dHA6Ly9jYWNlcnRzLmRp
# Z2ljZXJ0LmNvbS9EaWdpQ2VydFNIQTJBc3N1cmVkSURUaW1lc3RhbXBpbmdDQS5j
# cnQwDQYJKoZIhvcNAQELBQADggEBAEgc3LXpmiO85xrnIA6OZ0b9QnJRdAojR6Or
# ktIlxHBZvhSg5SeBpU0UFRkHefDRBMOG2Tu9/kQCZk3taaQP9rhwz2Lo9VFKeHk2
# eie38+dSn5On7UOee+e03UEiifuHokYDTvz0/rdkd2NfI1Jpg4L6GlPtkMyNoRdz
# DfTzZTlwS/Oc1np72gy8PTLQG8v1Yfx1CAB2vIEO+MDhXM/EEXLnG2RJ2CKadRVC
# 9S0yOIHa9GCiurRS+1zgYSQlT7LfySmoc0NR2r1j1h9bm/cuG08THfdKDXF+l7f0
# P4TrweOjSaH6zqe/Vs+6WXZhiV9+p7SOZ3j5NpjhyyjaW4emii8wggUxMIIEGaAD
# AgECAhAKoSXW1jIbfkHkBdo2l8IVMA0GCSqGSIb3DQEBCwUAMGUxCzAJBgNVBAYT
# AlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2Vy
# dC5jb20xJDAiBgNVBAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0x
# NjAxMDcxMjAwMDBaFw0zMTAxMDcxMjAwMDBaMHIxCzAJBgNVBAYTAlVTMRUwEwYD
# VQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xMTAv
# BgNVBAMTKERpZ2lDZXJ0IFNIQTIgQXNzdXJlZCBJRCBUaW1lc3RhbXBpbmcgQ0Ew
# ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQC90DLuS82Pf92puoKZxTlU
# KFe2I0rEDgdFM1EQfdD5fU1ofue2oPSNs4jkl79jIZCYvxO8V9PD4X4I1moUADj3
# Lh477sym9jJZ/l9lP+Cb6+NGRwYaVX4LJ37AovWg4N4iPw7/fpX786O6Ij4YrBHk
# 8JkDbTuFfAnT7l3ImgtU46gJcWvgzyIQD3XPcXJOCq3fQDpct1HhoXkUxk0kIzBd
# vOw8YGqsLwfM/fDqR9mIUF79Zm5WYScpiYRR5oLnRlD9lCosp+R1PrqYD4R/nzEU
# 1q3V8mTLex4F0IQZchfxFwbvPc3WTe8GQv2iUypPhR3EHTyvz9qsEPXdrKzpVv+T
# AgMBAAGjggHOMIIByjAdBgNVHQ4EFgQU9LbhIB3+Ka7S5GGlsqIlssgXNW4wHwYD
# VR0jBBgwFoAUReuir/SSy4IxLVGLp6chnfNtyA8wEgYDVR0TAQH/BAgwBgEB/wIB
# ADAOBgNVHQ8BAf8EBAMCAYYwEwYDVR0lBAwwCgYIKwYBBQUHAwgweQYIKwYBBQUH
# AQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wQwYI
# KwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFz
# c3VyZWRJRFJvb3RDQS5jcnQwgYEGA1UdHwR6MHgwOqA4oDaGNGh0dHA6Ly9jcmw0
# LmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwOqA4oDaG
# NGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RD
# QS5jcmwwUAYDVR0gBEkwRzA4BgpghkgBhv1sAAIEMCowKAYIKwYBBQUHAgEWHGh0
# dHBzOi8vd3d3LmRpZ2ljZXJ0LmNvbS9DUFMwCwYJYIZIAYb9bAcBMA0GCSqGSIb3
# DQEBCwUAA4IBAQBxlRLpUYdWac3v3dp8qmN6s3jPBjdAhO9LhL/KzwMC/cWnww4g
# Qiyvd/MrHwwhWiq3BTQdaq6Z+CeiZr8JqmDfdqQ6kw/4stHYfBli6F6CJR7Euhx7
# LCHi1lssFDVDBGiy23UC4HLHmNY8ZOUfSBAYX4k4YU1iRiSHY4yRUiyvKYnleB/W
# CxSlgNcSR3CzddWThZN+tpJn+1Nhiaj1a5bA9FhpDXzIAbG5KHW3mWOFIoxhynmU
# fln8jA/jb7UBJrZspe6HUSHkWGCbugwtK22ixH67xCUrRwIIfEmuE7bhfEJCKMYY
# Vs9BNLZmXbZ0e/VWMyIvIjayS6JKldj1po5SMIIFWTCCBEGgAwIBAgIQPXjX+XZJ
# YLJhffTwHsqGKjANBgkqhkiG9w0BAQsFADCByjELMAkGA1UEBhMCVVMxFzAVBgNV
# BAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQLExZWZXJpU2lnbiBUcnVzdCBOZXR3
# b3JrMTowOAYDVQQLEzEoYykgMjAwNiBWZXJpU2lnbiwgSW5jLiAtIEZvciBhdXRo
# b3JpemVkIHVzZSBvbmx5MUUwQwYDVQQDEzxWZXJpU2lnbiBDbGFzcyAzIFB1Ymxp
# YyBQcmltYXJ5IENlcnRpZmljYXRpb24gQXV0aG9yaXR5IC0gRzUwHhcNMTMxMjEw
# MDAwMDAwWhcNMjMxMjA5MjM1OTU5WjB/MQswCQYDVQQGEwJVUzEdMBsGA1UEChMU
# U3ltYW50ZWMgQ29ycG9yYXRpb24xHzAdBgNVBAsTFlN5bWFudGVjIFRydXN0IE5l
# dHdvcmsxMDAuBgNVBAMTJ1N5bWFudGVjIENsYXNzIDMgU0hBMjU2IENvZGUgU2ln
# bmluZyBDQTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAJeDHgAWryyx
# 0gjE12iTUWAecfbiR7TbWE0jYmq0v1obUfejDRh3aLvYNqsvIVDanvPnXydOC8KX
# yAlwk6naXA1OpA2RoLTsFM6RclQuzqPbROlSGz9BPMpK5KrA6DmrU8wh0MzPf5vm
# wsxYaoIV7j02zxzFlwckjvF7vjEtPW7ctZlCn0thlV8ccO4XfduL5WGJeMdoG68R
# eBqYrsRVR1PZszLWoQ5GQMWXkorRU6eZW4U1V9Pqk2JhIArHMHckEU1ig7a6e2iC
# Me5lyt/51Y2yNdyMK29qclxghJzyDJRewFZSAEjM0/ilfd4v1xPkOKiE1Ua4E4bC
# G53qWjjdm9sCAwEAAaOCAYMwggF/MC8GCCsGAQUFBwEBBCMwITAfBggrBgEFBQcw
# AYYTaHR0cDovL3MyLnN5bWNiLmNvbTASBgNVHRMBAf8ECDAGAQH/AgEAMGwGA1Ud
# IARlMGMwYQYLYIZIAYb4RQEHFwMwUjAmBggrBgEFBQcCARYaaHR0cDovL3d3dy5z
# eW1hdXRoLmNvbS9jcHMwKAYIKwYBBQUHAgIwHBoaaHR0cDovL3d3dy5zeW1hdXRo
# LmNvbS9ycGEwMAYDVR0fBCkwJzAloCOgIYYfaHR0cDovL3MxLnN5bWNiLmNvbS9w
# Y2EzLWc1LmNybDAdBgNVHSUEFjAUBggrBgEFBQcDAgYIKwYBBQUHAwMwDgYDVR0P
# AQH/BAQDAgEGMCkGA1UdEQQiMCCkHjAcMRowGAYDVQQDExFTeW1hbnRlY1BLSS0x
# LTU2NzAdBgNVHQ4EFgQUljtT8Hkzl699g+8uK8zKt4YecmYwHwYDVR0jBBgwFoAU
# f9Nlp8Ld7LvwMAnzQzn6Aq8zMTMwDQYJKoZIhvcNAQELBQADggEBABOFGh5pqTf3
# oL2kr34dYVP+nYxeDKZ1HngXI9397BoDVTn7cZXHZVqnjjDSRFph23Bv2iEFwi5z
# uknx0ZP+XcnNXgPgiZ4/dB7X9ziLqdbPuzUvM1ioklbRyE07guZ5hBb8KLCxR/Md
# oj7uh9mmf6RWpT+thC4p3ny8qKqjPQQB6rqTog5QIikXTIfkOhFf1qQliZsFay+0
# yQFMJ3sLrBkFIqBgFT/ayftNTI/7cmd3/SeUx7o1DohJ/o39KK9KEr0Ns5cF3kQM
# Ffo2KwPcwVAB8aERXRTl4r0nS1S+K4ReD6bDdAUK75fDiSKxH3fzvc1D1PFMqT+1
# i4SvZPLQFCExggRpMIIEZQIBATCBkzB/MQswCQYDVQQGEwJVUzEdMBsGA1UEChMU
# U3ltYW50ZWMgQ29ycG9yYXRpb24xHzAdBgNVBAsTFlN5bWFudGVjIFRydXN0IE5l
# dHdvcmsxMDAuBgNVBAMTJ1N5bWFudGVjIENsYXNzIDMgU0hBMjU2IENvZGUgU2ln
# bmluZyBDQQIQcihhzDecMy1dmXpidSGfHzAJBgUrDgMCGgUAoHgwGAYKKwYBBAGC
# NwIBDDEKMAigAoAAoQKAADAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgor
# BgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAjBgkqhkiG9w0BCQQxFgQUeYD2xVq4
# g2nFpkKnbLVnsoL0AZcwDQYJKoZIhvcNAQEBBQAEggEAhXN0JOUqDctn0Z44rXMO
# sAm3pNMDr1NxUBlQ3WMfDTIZs1DGn1+jLUhna3/FrtbGYbC/LL02E8uUnAzXCZ9+
# XFzuCpkZXhczv8rkUR0//q1vjwcIroq4Q1UP40k+NvcsMGgJOcOMMvN/IO8qKklA
# TB4ADOUyraY9j2MuIcJVNKFoIEyFYfaWqU/1lEH5+878zveZhOCg7gzmuSR6FHA6
# fQQXGLUqlxY0WSFBe29Q2T2DYd0C6hxrs3xZTKxALOPiRulEttE+EY68pjPORYCw
# +4uQQZYZjMmU63R4Q2f5jsUMeUhgQ1xd+TBzt52EBTX8sDiNe9SOjcyymb6zm7Ad
# YqGCAjAwggIsBgkqhkiG9w0BCQYxggIdMIICGQIBATCBhjByMQswCQYDVQQGEwJV
# UzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQu
# Y29tMTEwLwYDVQQDEyhEaWdpQ2VydCBTSEEyIEFzc3VyZWQgSUQgVGltZXN0YW1w
# aW5nIENBAhANQkrgvjqI/2BAIc4UAPDdMA0GCWCGSAFlAwQCAQUAoGkwGAYJKoZI
# hvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0BCQUxDxcNMjEwNzMxMTQyMTM2
# WjAvBgkqhkiG9w0BCQQxIgQgEAbm+O2CwZFo3aZHu1lPRnLanKwhLXEfaG1AooU8
# E9QwDQYJKoZIhvcNAQEBBQAEggEAe/4b2p9bRc+jWp+IOR8VaWtqi/FStWcQ3g7G
# Wdm1oVCvUsQcBcBXes60iwmj91g6wiAM6BVinZJwd8M8RQrCcFmcxTacJPp60+BJ
# 4v0b3EEWZhdY0pwQ1AeKulW4YybNkJVU9Gups00cggNULONkUkhKQoxcwp0cgpM7
# 2HDj0rhk1Ln0b1gBn0vx4VMeH3HJSUHww0kex1zCjbOGrU/49CVQ8IHQVI+aPW/M
# 5C3POVlBMTABpww6KCAOiCqjnSf+kivBXfMo881tJouhpTzw9tm8U9RZt3RiWeVf
# F7C/psd/r9EoaNb6Bi2M64Ox6sMnelKyZFXGQQHI8g11JddHnQ==
# SIG # End signature block
